#!/usr/bin/perl
#
#   orarman.pl - makes an online backup of an oracle database instance
#
# ---------------------------------------------------------
# Copyright 2008-2013, roveda
#
# This file is part of ORACLE_TOOLS.
#
# ORACLE_TOOLS is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# ORACLE_TOOLS is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ORACLE_TOOLS.  If not, see <http://www.gnu.org/licenses/>.
#
#
# ---------------------------------------------------------
# Synopsis:
#   perl orarman.pl <configuration file>
#
# ---------------------------------------------------------
# Description:
#   This script executes an RMAN backup of a running Oracle database. 
#
# ---------------------------------------------------------
# Dependencies:
#   Misc.pm
#   Uls2.pm
#   uls-client-2.0-1 or later
#   You must set the necessary environment variables for
#   used operating system commands before starting this script.
#
# ---------------------------------------------------------
# Disclaimer:
#   The script has been tested and appears to work as intended,
#   but there is no guarantee that it behaves as YOU expect.
#   You should always run new scripts on a test instance initially.
#
# ---------------------------------------------------------
# Versions:
#
# "@(#)  orarman.pl   0.01   2008-05-28   christvo"
#        spin off from orabackup.pl
# "@(#)  orarman.pl   0.02   2008-06-12   roveda"
#        Changed the CONFIGURE CONTROLFILE...FORMAT.
# "@(#)  orarman.pl   0.03   2008-11-13   roveda"
#        Ported to Uls2.pm. Added RMAN commands.
# "@(#)  orarman.pl   0.04   2009-01-30   roveda"
#        Minor changes to documentation below __END__.
# "@(#)  orarman.pl   0.05   2009-02-16   roveda"
#        Changed to Uls2.pm, only archive logging
# "@(#)  orarman.pl   0.06   2009-12-30   roveda"
#        Now removing old temporary log files correctly.
# "@(#)  orarman.pl   0.07   2010-01-29   roveda"
#        Changed the RMAN commands (DELETE OBSOLETE instead of 
#        DELETE ARCHIVELOG and DELETE EXPIRED BACKUP).
# "@(#)  orarman.pl   0.08   2010-02-26   roveda"
#        MAX_SET_SIZE can now be set in the configuration file, the
#        default is now UNLIMITED (was 30G).
# "@(#)  orarman.pl   0.09   2010-03-04   roveda"
#        Added the COMPRESSED feature, changed the REDUNDANCY feature.
# "@(#)  orarman.pl   0.11   2010-09-20   roveda"
#        Moved from only recovery window to retention policy for 
#        recovery window and redundancy.
# "@(#)  orarman.pl   0.12   2010-12-22   roveda"
#        Backup to fast recovery area now possible. But removed the 
#        copying of the listener.ora, tnsnames.ora and sqlnet.ora.
# "@(#)  orarman.pl   0.13   2011-11-11   roveda"
#        Added the GPL.
# "@(#)  orarman.pl   0.14   2012-08-30   roveda"
#        Changed to ULS-modules.
# "@(#)  orarman.pl   0.15   2013-03-29   roveda"
#        Changed to new configuration file format. The complete 
#        RMAN command must now be specified in the configuration file.
#        No substitutions.
# "@(#)  orarman.pl   0.16   2013-08-17   roveda"
#        Modifications to match the new single configuration file.
#
#
#        Change also $VERSION later in this script!
#
# ===================================================================


use 5.8.0;
use strict;
use warnings;
use File::Basename;
use File::Copy;

# These are ULS-modules:
use lib ".";
use Misc 0.32;
use Uls2 1.10;

my $VERSION = 0.16;

# ===================================================================
# The "global" variables
# ===================================================================

# Keeps the name of this script.
# $CURRPROG = basename($0, ".pl");   # extension is removed
my $CURRPROG = basename($0);

my $currdir = dirname($0);
my $start_secs = time;

# The runtime of this script is measured in minutes
my $RUNTIME_UNIT = "M";

my $WORKFILEPREFIX;
my $TMPOUT1;
my $TMPOUT2;
my $TMPOUT3;
my $LOCKFILE;
my $DELIM = "!";

# The $MSG will contain still the "OK", when reaching the end
# of the script. If any errors occur (which the script is testing for)
# the $MSG will contain "ERROR" or a complete error message, additionally,
# the script will send any error messages to the uls directly.
# <hostname> - $ULS_SECTION - __<IDENTIFIER> - message
my $MSG = "OK";

# This hash keeps the documentation for the teststeps.
my %TESTSTEP_DOC;

# Keeps the contents of the configuration file
my %CFG;

# This keeps the settings for the ULS
my %ULS;

# Holds the __$CURRPROG or $CFG{"IDENTIFIER"} just for easy usage.
my $IDENTIFIER;

# Destination directory of the backup
my $BACKUP_DESTINATION = "";

# The default command to execute sql commands.
my $SQLPLUS_COMMAND = 'sqlplus -S "/ as sysdba"';

# The default command to execute sql commands.
# NOTE: there is another rman in /usr/X11R6/bin!!!
#
# my $RMAN_INVOCATION = '/oracle/admin/product/10.2.0/db_1/bin/rman TARGET / NOCATALOG';
my $RMAN_INVOCATION = $ENV{ORACLE_HOME} . '/bin/rman TARGET / ';

# A parameter is given on the command line that specifies
# the RMAN command within the section ORARMAN to be used.

my $ORARMAN_COMMAND = "";



# ===================================================================
# The subroutines
# ===================================================================

sub output_error_message {
  # output_error_message(<message>)
  #
  # Send the given message, set the $MSG variable and
  # print out the message.

  $MSG = "ERROR";

  foreach my $msg (@_) {
    print STDERR "$msg\n";
    uls_value($IDENTIFIER, "message", $msg, " ");
  }

} # output_error_message


# ------------------------------------------------------------
sub errors_in_file {
  # errors_in_file <filename>
  #
  # Check contents of e.g. $TMPOUT1 for ORA- errors.

  my $ret = 0;
  my $filename = $_[0];

  if (! open(INFILE, "<$filename")) {
    output_error_message(sub_name() . ": Error: Cannot open '$filename' for reading. $!");
    $ret = 1;
    return($ret);
  }

  my $L;

  while ($L = <INFILE>) {
    chomp($L);
    if ($L =~ /ORA-\d+/) {
      # yes, there have been errors.
      output_error_message(sub_name() . ": Error: There have been error(s) in file '$filename'!");
      $ret = 1;
    }

    if ($L =~ /RMAN-\d+/) {
      # yes, there have been errors.
      output_error_message(sub_name() . ": Error: There have been error(s) in file '$filename'!");
      $ret = 1;
    }

    if ($L =~ /error/i) {
      # yes, there have been errors.
      output_error_message(sub_name() . ": Error: There have been error(s) in file '$filename'!");
      $ret = 1;
    }

  } # while

  if (! close(INFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$filename'. $!");
    $ret = 1;
  }
  return($ret);
} # errors_in_file


# ------------------------------------------------------------
sub reformat_spool_file {
  # reformat_spool_file(<filename>)
  #
  # Reformats the spool file, removes unnecessary blanks surrounding
  # the delimiter, like this:
  #
  # ARTUS                         !          2097152000!            519569408
  # SYSTEM                        !          2097152000!            174129152
  # UNDOTS                        !          1048576000!             10027008
  #
  # ARTUS!2097152000!519569408
  # SYSTEM!2097152000!174129152
  # UNDOTS!1048576000!10027008
  #
  # This is necessary, because matching of constant expressions (like 'ARTUS')
  # would fail (the proper expression would be: 'ARTUS                         ').

  my $filename = $_[0];
  my $tmp_filename = "$filename.tmp";

  if (! open(INFILE, $filename)) {
    output_error_message(sub_name() . ": Error: Cannot open '$filename' for reading. $!");
    return(0);
  }

  if (! open(OUTFILE, ">$tmp_filename")) {
    output_error_message(sub_name() . ": Error: Cannot open '$tmp_filename' for writing. $!");
    return(0);
  }

  my $L;

  while($L = <INFILE>) {
    chomp($L);
    my @e = split($DELIM, $L);
    my $E;
    foreach $E(@e) {
      print OUTFILE trim($E), $DELIM;
    }
    print OUTFILE "\n";
  }

  if (! close(INFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$filename'. $!");
    return(0);
  }

  if (! close(OUTFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$tmp_filename'. $!");
    return(0);
  }

  if (! copy($tmp_filename, $filename)) {
    output_error_message(sub_name() . ": Error: Cannot copy '$tmp_filename' to '$filename'. $!");
    return(0);
  }

  if (! unlink($tmp_filename)) {
    output_error_message(sub_name() . ": Error: Cannot remove '$tmp_filename'. $!");
    return(0);
  }
} # reformat_spool_file


# ------------------------------------------------------------
sub exec_sql {
  # <sql command>
  # Just executes the given sql statement against the current database instance.
  # If <verbose> is a true expression (e.g. a 1) the sql statement will
  # be printed to stdout.

  # connect / as sysdba

  my $sql = "
    set newpage 0
    set space 0
    set linesize 10000
    set pagesize 0
    set echo off
    set feedback off
    set heading off
    set markup html off spool off

    set numwidth 20
    set colsep '$DELIM'

    spool $TMPOUT1;

    $_[0]

    spool off;
  ";

  print "\nexec_sql(): SQL:\n";
  print "$sql\n";

  if (! open(CMDOUT, "| $SQLPLUS_COMMAND")) {
    output_error_message(sub_name() . ": Error: Cannot open pipe to sqlplus. $!");
    return(0);   # error
  }
  print CMDOUT "$sql\n";
  if (! close(CMDOUT)) {
    output_error_message(sub_name() . ": Error: Cannot close pipe to sqlplus. $!");
    return(0);
  }

  reformat_spool_file($TMPOUT1);

  return(1);   # ok
} # exec_sql



# -------------------------------------------------------------------
sub do_sql {
  # do_sql(<sql>)
  #
  # Returns 0, when errors have occurred,
  # and outputs an error message,
  # returns 1, when no errors have occurred.

  if (exec_sql($_[0])) {
    if (errors_in_file($TMPOUT1)) {
      output_error_message(sub_name() . ": Error: there have been errors when executing the sql statement.");
      uls_send_file_contents($IDENTIFIER, "message", $TMPOUT1);
      return(0);
    }
    # Ok
    return(1);
  }

  output_error_message(sub_name() . ": Error: Cannot execute sql statement.");
  uls_send_file_contents($IDENTIFIER, "message", $TMPOUT1);

  return(0);

} # do_sql


# -------------------------------------------------------------------
sub clean_up {
  # clean_up(<file list>)
  #
  # Remove all left over files at script end.

  title("Cleaning up");

  # Remove temporary files.
  foreach my $file (@_) {
    if (-e $file) {
      print "Removing temporary file '$file' ...";
      if (unlink($file)) {print "Done.\n"}
      else {print "Failed.\n"}
    }
  }

} # clean_up


# -------------------------------------------------------------------
sub send_runtime {
  # The runtime of this script
  # send_runtime(<start_secs> [, {"s"|"m"|"h"}]);

  # Current time minus start time.
  my $rt = time - $_[0];

  my $unit = "S";
  if ($_[1]) {$unit = uc($_[1])}

  if    ($unit eq "M") { uls_value($IDENTIFIER, "runtime", pround($rt / 60.0, -1), "min") }
  elsif ($unit eq "H") { uls_value($IDENTIFIER, "runtime", pround($rt / 60.0 / 60.0, -2), "h") }
  else                 { uls_value($IDENTIFIER, "runtime", pround($rt, 0), "s") }

} # send_runtime


# ------------------------------------------------------------
sub send_doc {
  # send_doc(<title> [, <as title>])
  #
  # If the <title> is found in the $TESTSTEP_DOC hash, then
  # the associated text is sent as documentation to the ULS.
  # Remember: the teststep must exist in the ULS before any
  #           documentation can be saved for it.
  # If the alias <as title> is given, the associated text is
  # sent to the ULS for teststep <as title>. So you may even
  # document variable teststeps with constant texts. You may
  # substitute parts of the contents of the hash value, before
  # it is sent to the ULS.

  my $title = $_[0];
  my $astitle = $title;

  if ($_[1]) {$astitle = $_[1]}

  if (%TESTSTEP_DOC) {
    if ($TESTSTEP_DOC{$title}) {
      # TODO: You may want to substitute <title> with <astitle> in the text?
      uls_doc($astitle, $TESTSTEP_DOC{$title})
    } else {
      print "No documentation for '$title' found.\n";
    }
  }

} # send_doc


# -------------------------------------------------------------------
sub general_info {
  # Gather some general info about the current oracle instance
  # This is only used to check, whether Oracle Database Server
  # is running (OPEN).

  # This sub returns a value, whether the rest of the script is
  # executed or not.

  title("Checking for running Oracle instance");

  my $sql = "select 'database status', status from v\$instance;";

  if (! do_sql($sql)) {return(0)}

  my $V = trim(get_value($TMPOUT1, $DELIM, "database status"));
  if ($V ne "OPEN") {
    output_error_message(sub_name() . ": Error: Database instance is not OPEN, but '$V'.");
    return(0);
  } else {
    print "Database is OPEN => ok.\n";
  }
  # uls_value($ts, "database status", $V, " ");

  # send_doc($ts);

  return(1); # ok
} # general_info



# -------------------------------------------------------------------
sub check_for_archivelog {
  title("Checking for required ARCHIVELOG mode");

  my $sql = "select 'log mode', log_mode from v\$database;";

  # 'LOGMODE LOG_MODE
  # -------- ------------
  # log mode NOARCHIVELOG

  if (! do_sql($sql)) {return(0)}

  my $V = uc(trim(get_value($TMPOUT1, $DELIM, "log mode")));
  # Keep in mind: it may be ARCHIVELOG or NOARCHIVELOG. A pattern
  # matching must cope with that.
  if ($V ne "ARCHIVELOG") {
    output_error_message(sub_name() . ": Error: Database instance is not in ARCHIVELOG mode but '$V' => no online backup possible.");
    return(0);
  } else {
    print "Database instance is in ARCHIVELOG mode => ok.\n";
  }

  return(1); # ok
} # check_for_archivelog



#--------------------------------------------------------------------
sub exec_system {

  my $cmd = $_[0];

  my $ret = 1;  # assume success

  print "$cmd\n";

  system($cmd);

  if ($? == -1) {

    output_error_message(sub_name() . ": Error: failed to execute '$cmd': $!");
    $ret = 0;

  } elsif ($? & 127) {

    my $txt = sprintf("child died with signal %d, %s coredump", ($? & 127),  ($? & 128) ? 'with' : 'without');
    output_error_message(sub_name() . ": Error: $txt");
    $ret = 0;

  } else {
    # Ok

    my $txt = sprintf("child exited with value %d", $? >> 8);
    print "$txt\n";
    $ret = 1;  # Ok

  }

  return($ret);

} # exec_system


#--------------------------------------------------------------------
sub grep_file {
  # grep_file(<file>, <pattern>)

  my ($file, $patt) = @_;

  my $ret = 0;

  if (! open(RD, "<", $file)) {
    output_error_message(sub_name() . ": Error: Cannot open '$file' for reading. $!");
    return($ret);
  }

  while (my $L = <RD>) {
    if ($L =~ /$patt/i) {
      $ret = 1;
      last;
    }
  }

  if (! close(RD)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$file'. $!");
    $ret = 0;
  }

  return($ret);

} # grep_file


#--------------------------------------------------------------------
sub exec_rman {
  # exec_rman(<command file>);

  # Just executes the given rman command file against the rman.

  # rman cmdfile 'xyz.rman' checksyntax log '/tmp/rman_syntax_??.log'
  # ...
  # The cmdfile has no syntax errors
  # ...
  #
  # rman target / cmdfile 'xyz.rman' log 'sasasa'

  my $command_file = $_[0];

  # print "command_file = $command_file\n";

  my $rman_command = $CFG{"ORACLE.RMAN_INVOCATION"} || $RMAN_INVOCATION;
  print "RMAN invocation command: $rman_command\n";

  # -----
  # check syntax

  my $cmd = "$rman_command cmdfile '$command_file' checksyntax log '$TMPOUT1'";

  # print "cmd=$cmd\n";

  if (! exec_system($cmd)) { return(0) }

  # Check the output
  # (only in English? or also other languages possible?)
  if (! grep_file($TMPOUT1, "The cmdfile has no syntax errors")) {
    output_error_message(sub_name() . ": Error: Improper syntax.");
    uls_file({
      teststep => $IDENTIFIER
     ,detail   => "RMAN syntax check log file"
     ,filename => $TMPOUT1
     ,rename_to => "rman_syntax_check_log_file.txt"
    });
    return(0);
  }

  $cmd = "$rman_command cmdfile '$command_file' log '$TMPOUT3'";

  my $ret = exec_system($cmd);

  # Send the command file.
  uls_file({
    teststep => $IDENTIFIER
   ,detail   => "RMAN command file"
   ,filename => $command_file
   ,rename_to => "rman_command_file.sql"
  });

  # Send the log file.
  uls_file({
    teststep => $IDENTIFIER
   ,detail   => "RMAN log file"
   ,filename => $TMPOUT3
   ,rename_to => "rman_log_file.txt"
  });

  if (! $ret) { return(0) }

  # Check the output
  if (grep_file($TMPOUT3, 'ORA-\d+')) {
    output_error_message(sub_name() . ": Error: when executing rman command file.");
    return(0);
  }

  if (grep_file($TMPOUT3, 'RMAN-\d+')) {
    output_error_message(sub_name() . ": Error: when executing rman command file.");
    return(0);
  }

  # This check may be a bit too general
  # if (grep_file($TMPOUT3, "error")) {
  #   output_error_message(sub_name() . ": Error: when executing rman command file.");
  #   return(0);
  # }

  return(1);

} # exec_rman


#--------------------------------------------------------------------
sub rman_backup {

  title("RMAN backup");

  # -----
  # The command sequence from the configuration file for the RMAN backup action.

  my $C = $CFG{"ORARMAN.$ORARMAN_COMMAND"};
  print "\n";
  print "RMAN command to execute:\n";
  print "$C\n\n";

  if (! $C) {
    output_error_message(sub_name() . ": Error: No matching RMAN command found in configuration file for parameter '$ORARMAN_COMMAND'.");
    return(0);
  }

  # -----
  # Output to temporary command file

  if (! open(OUTFILE, ">", $TMPOUT2)) {
    output_error_message(sub_name() . ": Error: Cannot open '$TMPOUT2' for writing. $!");
    return(0);
  }

  print OUTFILE "$C\n";

  if (! close(OUTFILE)) {
    output_error_message(sub_name() . ": Error: Cannot close file handler for file '$TMPOUT2'. $!");
    return(0);
  }

  # -----
  # Execute the command file

  exec_rman($TMPOUT2);

  print "Backup with rman done.\n";

  return(1);

} # rman_backup



# ===================================================================
# main
# ===================================================================
#
# initial customization, no output should happen before this.
# The environment must be set up already.

$CURRPROG = basename($0);
$IDENTIFIER = "_" . basename($0, ".pl");

my $initdir = $ENV{"TMP"} || $ENV{"TEMP"} || $currdir;
my $initial_logfile = "${currdir}/${CURRPROG}_$$.tmp";

# re-direct stdout and stderr to a temporary logfile.
open(STDOUT, "> $initial_logfile") or die "Cannot re-direct STDOUT to $initial_logfile.\n";
open(STDERR, ">&STDOUT") or die "Cannot re-direct STDERR to STDOUT.\n";
select(STDERR);
$| = 1;
select(STDOUT);
$| = 1;           # make unbuffered

# -------------------------------------------------------------------
# From here on, STDOUT+ERR is logged.

title("Start");
print "$CURRPROG is started in directory '$currdir'\n";

# ----------
# Get the configuration file.

# first command line argument
my $cfgfile = $ARGV[0];
print "Configuration file=$cfgfile\n";

if (! get_config2($cfgfile, \%CFG, "GENERAL", "ORACLE", "ULS", "ORARMAN")) {
  print STDERR $CURRPROG . ": Error: Cannot parse configuration file '$cfgfile' correctly!\n";
  exit(1);
}

print "-- Effective configuration:\n";
show_hash(\%CFG, " = ");
print "-----\n\n";

# -----
# RMAN command identifier
# (like ORARMAN_FULL or LEVEL0)
# Must be a parameter of the [ORARMAN] section in the oracle_tools.conf

# second command line argument
$ORARMAN_COMMAND = $ARGV[1];
if (! $ORARMAN_COMMAND) {
  print STDERR $CURRPROG . ": Error: no command line argument given for an ORARMAN parameter!\n";
  exit(2);
}
print "RMAN command parameter:$ORARMAN_COMMAND\n\n";

# ----------
# This sets the %ULS to all necessary values
# deriving from %CFG (configuration file),
# environment variables (ULS_*) and defaults.

uls_settings(\%ULS, \%CFG);

print "-- ULS settings:\n";
show_hash(\%ULS, " = ");
print "-----\n\n";

# ----------
# IDENTIFIER

# Set default
$IDENTIFIER = $CFG{"ORARMAN.IDENTIFIER"} || $IDENTIFIER;

# Add the orarman command parameter to the identifier,
# that allows a monitoring distinction (especially for isAlives) between the different
# orarman executions. Normally, there is
# ONE full backup each week (level0),
# ONE cumulative incremental backup each day (level1) and
# ONE backup of all not yet backed up archived redo logs each hour (redologs).
# You SHOULD have used ':ORARMAN_PARAMETER:' in the IDENTIFIER parameter of the ORARMAN section!

$IDENTIFIER =~ s/:ORARMAN_PARAMETER:/lc($ORARMAN_COMMAND)/eg;
print "IDENTIFIER=$IDENTIFIER\n";
# From here on, you may use $IDENTIFIER for uniqueness

# -------------------------------------------------------------------
# environment
#
# Check here the necessary environment variables

if ((! $ENV{"ORACLE_SID"})  && $CFG{"ORACLE.ORACLE_SID"})  {$ENV{"ORACLE_SID"}  = $CFG{"ORACLE.ORACLE_SID"}}
if ((! $ENV{"ORACLE_HOME"}) && $CFG{"ORACLE.ORACLE_HOME"}) {$ENV{"ORACLE_HOME"} = $CFG{"ORACLE.ORACLE_HOME"}}

if (! $ENV{"ORACLE_SID"}) {
  print STDERR "$CURRPROG: Error: ORACLE_SID is not set in the environment => aborting.\n";
  exit(1);
}
if (! $ENV{"ORACLE_HOME"}) {
  print STDERR "$CURRPROG: Error: ORACLE_HOME is not set in the environment => aborting.\n";
  exit(1);
}
print "Oracle environment variables:\n";
print "ORACLE_HOME=", $ENV{"ORACLE_HOME"}, "\n";
print "ORACLE_SID=", $ENV{"ORACLE_SID"}, "\n";
print "\n";

# -------------------------------------------------------------------
# Working directory

my $workdir = $ENV{"WORKING_DIR"} || $CFG{"GENERAL.WORKING_DIR"} || $currdir;

if ( ! (-e $workdir)) {
  print "Creating directory '$workdir' for work files.\n";
  if (! mkdir($workdir)) {
    print STDERR "$CURRPROG: Error: Cannot create directory '$workdir' => aborting!\n";
    exit(1);
  }
}

$WORKFILEPREFIX = "${workdir}/${IDENTIFIER}";
# If no oracle sid is found in the workfile prefix, then add it.
# Normally, this should be set in the configuration file.
# if ($WORKFILEPREFIX !~ /$ENV{"ORACLE_SID"}/) { $WORKFILEPREFIX .= "_" . $ENV{"ORACLE_SID"} }
print "WORKFILEPREFIX=$WORKFILEPREFIX\n";

# -------------------------------------------------------------------
# Setting up a lock file to prevent more than one instance of this
# script starting simultaneously.

$LOCKFILE = "${WORKFILEPREFIX}.LOCK";
print "LOCKFILE=$LOCKFILE\n";

if (! lockfile_build($LOCKFILE)) {
  # LOCK file exists and process is still running, abort silently.
  print "Another instance of this script is still running => aborting!\n";
  exit(1);
}

# -------------------------------------------------------------------
# The final log file.

my $logfile = "$WORKFILEPREFIX.log";

move_logfile($logfile);

# re-direct stdout and stderr to a logfile.
open(STDOUT, "> $logfile") or die "Cannot re-direct STDOUT to $logfile. $!\n";
open(STDERR, ">&STDOUT") or die "Cannot re-direct STDERR to STDOUT. $!\n";
select(STDERR);
$| = 1;
select(STDOUT);
$| = 1;           # make unbuffered

# Copy initial logfile contents to current logfile.
if (-e $initial_logfile) {
  print "Contents of initial logfile '$initial_logfile':\n";
  open(INITLOG, $initial_logfile);
  while (<INITLOG>) {print;}
  close(INITLOG);
  print "Removing initial log file '$initial_logfile'.\n";
  my $c = unlink($initial_logfile);
  print "$c file(s) successfully removed.\n";

  print "Remove possible old temporary files.\n";
  # Remove old .tmp files
  opendir(INITDIR, $initdir);
  my @files = grep(/$CURRPROG.*\.tmp/, map("$initdir/$_", readdir(INITDIR)));
  foreach my $file (@files) {
    # Modification time of file, also fractions of days.
    my $days = pround(-M $file, -1);

    if ($days > 5) {
      print "Remove '", basename($file), "', ($days days old)...\n";
      my $c = unlink($file);
      print "$c file(s) successfully removed.\n";
    }
  } # foreach
}

# -------------------------------------------------------------------
title("Set up ULS");

# Initialize uls with basic settings
uls_init(\%ULS);

my $d = iso_datetime($start_secs);
$d =~ s/\d{1}$/0/;

set_uls_timestamp($d);


# ---- Send name of this script and its version
uls_value($IDENTIFIER, "script name, version", "$CURRPROG, $VERSION", " ");

# ---- Send also versions of ULS-modules.
uls_value($IDENTIFIER, "modules", "Misc $Misc::VERSION, Uls2 $Uls2::VERSION", " ");

uls_timing({
    teststep  => $IDENTIFIER
  , detail    => "start-stop"
  , start     => iso_datetime($start_secs)
});

# -------------------------------------------------------------------
# The real work starts here.
# ------------------------------------------------------------

# Define some temporary file names
$TMPOUT1 = "${WORKFILEPREFIX}_1.tmp";
print "TMPOUT1=$TMPOUT1\n";

# Use for the rman commands
$TMPOUT2 = "${WORKFILEPREFIX}_2.tmp";
print "TMPOUT2=$TMPOUT2\n";

# Use for the rman output
$TMPOUT3 = "${WORKFILEPREFIX}_3.tmp";
print "TMPOUT3=$TMPOUT3\n";

print "DELIM=$DELIM\n";


# -----
# Which section is used in this run
uls_value($IDENTIFIER, "used parameter in configuration file", $ORARMAN_COMMAND, " ");


# ----- documentation -----
# Send the documentation to ULS

title("Documentation");

print "Prepare the documentation.\n";

# de-reference the return value to the complete hash.
%TESTSTEP_DOC = %{doc2hash(\*DATA)};


# ----- sqlplus command -----
# Check, if the sqlplus command has been redefined in the configuration file.

$SQLPLUS_COMMAND = $CFG{"ORACLE.SQLPLUS_COMMAND"} || $SQLPLUS_COMMAND;


# -------------------------------------------------------------------
# Check if Oracle database is running.

if (! general_info()) {
  output_error_message("$CURRPROG: Error: A fatal error has ocurred! Aborting script.");

  send_runtime($start_secs, $RUNTIME_UNIT);
  uls_timing($IDENTIFIER, "start-stop", "stop");
  uls_flush(\%ULS);

  clean_up($TMPOUT1, $TMPOUT2, $TMPOUT3, $LOCKFILE);
  exit(1);
}

# -------------------------------------------------------------------
# Check if Oracle database runs in ARCHIVELOG mode (else no online backup possible)

if (! check_for_archivelog()) {
  output_error_message("$CURRPROG: Error: A fatal error has ocurred! Aborting script.");

  send_runtime($start_secs, $RUNTIME_UNIT);
  uls_timing($IDENTIFIER, "start-stop", "stop");
  uls_flush(\%ULS);

  clean_up($TMPOUT1, $TMPOUT2, $TMPOUT3, $LOCKFILE);
  exit(1);
}


#--------------------------------------------------------------------

rman_backup();


# The real work ends here.
# -------------------------------------------------------------------

# Any errors will have sent already its error messages.
uls_value($IDENTIFIER, "message", $MSG, " ");

send_doc($CURRPROG, $IDENTIFIER);

send_runtime($start_secs, $RUNTIME_UNIT);

uls_timing($IDENTIFIER, "start-stop", "stop");
uls_flush(\%ULS);


# -------------------------------------------------------------------
clean_up($TMPOUT1, $TMPOUT2, $TMPOUT3, $LOCKFILE);
title("END");

if ($MSG eq "OK") {exit(0)}
else {exit(1)}


#########################
# end of script
#########################

__END__

# The format:
#
# *<teststep title>
# <any text>
# <any text>
#
# *<teststep title>
# <any text>
# <any text>
# ...
#
# Remember to keep the <teststep title> equal to those used
# in the script. If not, you won't get the expected documentation.

#########################
*orarman.pl
============

This is a backup script that runs RMAN against an Oracle database and performs a 
command that is defined in the first command line argument <configuration_file>. 
In that <configuration_file> the script searches for an RMAN 
<command_identifier> which is specified as second command line argument
and executes that without making any changes to that.

This script is part of the ORACLE_TOOLS and works best with the Universal Logging System (ULS). Visit the ULS homepage at http://www.universal-logging-system.org

message:
  If the script runs fine, it returns 'OK', else an error message.
  You should generate a notification for any other value.

runtime:
  The runtime of the script.

start-stop:
  The start and stop timestamps for the script.

script name, version:
  The name and version of this script.

modules:
  The used modules (.pm).

RMAN command file:
  Keeps the commands executed by the script.

RMAN log file:
  Keeps the log file of the execution of the RMAN commands.



Copyright 2008-2013, roveda

ORACLE_TOOLS is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

ORACLE_TOOLS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with ORACLE_TOOLS.  If not, see <http://www.gnu.org/licenses/>.

